<?php

declare(strict_types=1);

namespace Erlage\Photogram\Data\Dtos\Common;

use Erlage\Photogram\Data\Dtos\AbstractDTO;
use Erlage\Photogram\Data\Dtos\Traits\TraitComplaintValidatorDTO;
use Erlage\Photogram\Data\Dtos\Traits\TraitComplaintTransformerDTO;

class StringCollectionMapDTO extends AbstractDTO
{
    /**
     * to comply with DTO interface
     */
    use TraitComplaintValidatorDTO;
    use TraitComplaintTransformerDTO;

    /**
     * @var StringListDTO[] 
     */
    private $collections = array();

    /*
    |--------------------------------------------------------------------------
    | constructor without promoted properties, helps static analysis
    |--------------------------------------------------------------------------
    */

    /**
     * @param StringListDTO[] $entires 
     */
    public function __construct($entires = array())
    {
        foreach ($entires as $itemKey => $itemValue)
        {
            $this -> add($itemKey, $itemValue);
        }
    }

    /*
    |--------------------------------------------------------------------------
    | from json map
    |--------------------------------------------------------------------------
    */

    /**
     * @return static 
     */
    public static function fromJsonMap(array $data)
    {
        /**
         * @var StringListDTO[]
         */
        $collections = array();

        foreach ($data as $dataKey => $dataItem)
        {
            $collections[$dataKey] = StringListDTO::fromJsonMap($dataItem);
        }

        return new static($collections);
    }

    /*
    |--------------------------------------------------------------------------
    | interface
    |--------------------------------------------------------------------------
    */

    /**
     * whether post content contains any item
     * method simply runs count() call on items property and checks whether
     * count returns a non-negative number
     */
    public function isEmpty(): bool
    {
        return $this -> getLength() > 0;
    }

    public function containsKey(string $key): bool
    {
        return isset($this -> collections[$key]);
    }

    public function containsValue(string $value): bool
    {
        return \in_array($value, \array_values($this -> collections));
    }

    public function add(string $key, StringListDTO $value): void
    {
        $this -> collections[$key] = $value;
    }

    public function getValue(string $key): StringListDTO
    {
        return $this -> collections[$key];
    }

    public function updateValue(string $key, string $value): void
    {
        $this -> collections[$key] = $value;
    }

    public function getLength(): int
    {
        return \count($this -> collections);
    }

    /*
    |--------------------------------------------------------------------------
    | serialization for json_encode
    |--------------------------------------------------------------------------
    */

    public function jsonSerialize()
    {
        return $this -> collections;
    }

    /*
    |--------------------------------------------------------------------------
    | implement DTO methods
    |--------------------------------------------------------------------------
    */

    /**
     * @param StringCollectionMapDTO $stringCollectionListDTO
     */
    public static function serialize($stringCollectionListDTO): string
    {
        return \json_encode($stringCollectionListDTO);
    }

    /**
     * @return StringCollectionMapDTO 
     */
    public static function deSerialize(string $stringCollectionListDTO)
    {
        $data = \json_decode($stringCollectionListDTO, true);

        return static::fromJsonMap($data);
    }
}
